// Export field options
const exportFieldsDefault = {
    'name': true,
    'fulladdress': true,
    'street': true,
    'locality': true,
    'municipality': true,
    'postal code': true,
    'email': true,
    'phone': true,
    'business categories': true,
    'total reviews': true,
    'average rating': true,
    'latitude': true,
    'longitude': true,
    'website': true,
    'domain': true,
    'cid': true,
    'place_id': true,
    'google maps_url': true,
    'linkedin_url': true,
    'facebook_url': true,
    'twitter_url': true,
    'instagram_url': true,
    'youtube_url': true,
    'pinterest_url': true,
    'facebook pixel': true,
    'google tag manager': true,
    'google analytics': true,
    'opening hours': true,
    'price_level': true,
    'business_status': true,
    'photos': true
};

let exportFields = { ...exportFieldsDefault };
let optionsExpanded = false;

// DOM Elements
const statusCard = document.getElementById('statusCard');
const statusIcon = document.getElementById('statusIcon');
const statusTitle = document.getElementById('statusTitle');
const statusMessage = document.getElementById('statusMessage');
const emailModal = document.getElementById('emailModal');
const userEmailInput = document.getElementById('userEmail');
const saveEmailBtn = document.getElementById('saveEmailBtn');


const exportOptions = document.getElementById('exportOptions');
const optionsGrid = document.getElementById('optionsGrid');
const toggleOptions = document.getElementById('toggleOptions');
const clearBtn = document.getElementById('clearBtn');
const exportBtn = document.getElementById('exportBtn');

// Initialize
init();

async function init() {
    await loadExportFields();
    renderExportOptions();
    await checkUserIdentity();
    await checkStatus();
    setupEventListeners();
    startStatusMonitoring();
}

// Load export fields from storage
async function loadExportFields() {
    const result = await chrome.storage.local.get(['exportOptions']);
    if (result.exportOptions) {
        try {
            exportFields = JSON.parse(result.exportOptions);
        } catch (e) {
            exportFields = { ...exportFieldsDefault };
        }
    }
}

// Render export options checkboxes
function renderExportOptions() {
    optionsGrid.innerHTML = '';

    for (const [key, value] of Object.entries(exportFields)) {
        const wrapper = document.createElement('div');
        wrapper.className = 'option-checkbox';

        const checkbox = document.createElement('input');
        checkbox.type = 'checkbox';
        checkbox.id = `opt_${key}`;
        checkbox.checked = value;
        checkbox.addEventListener('change', (e) => updateExportField(key, e.target.checked));

        const label = document.createElement('label');
        label.htmlFor = `opt_${key}`;
        label.textContent = key;

        wrapper.appendChild(checkbox);
        wrapper.appendChild(label);
        optionsGrid.appendChild(wrapper);
    }
}

// Update export field preference
async function updateExportField(key, value) {
    exportFields[key] = value;
    await chrome.storage.local.set({ 'exportOptions': JSON.stringify(exportFields) });
}

// Check current status
async function checkStatus() {
    try {
        // Check if on Google Maps page
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        const isGoogleMaps = tab?.url?.includes('google.') && tab?.url?.includes('/maps');

        // Get stored data
        const result = await chrome.storage.local.get(['bg_businesses_temp', 'TF']);
        const businesses = result.bg_businesses_temp ? JSON.parse(result.bg_businesses_temp) : [];
        const totalFound = result.TF || 0;

        // Update UI

        if (businesses.length > 0) {
            showActiveStatus('Data Ready', `${businesses.length} businesses ready to export`);
            showButtons();
            exportOptions.style.display = 'block';


        } else if (isGoogleMaps) {
            showReadyStatus('On Google Maps', 'Search for businesses to start extracting');
        } else {
            showReadyStatus('Ready to Extract', 'Navigate to Google Maps and search for businesses');
        }
    } catch (error) {
        console.error('Error checking status:', error);
        showErrorStatus('Error', 'Failed to load data');
    }
}

// Show ready status
function showReadyStatus(title, message) {
    statusCard.className = 'status-card';
    statusTitle.textContent = title;
    statusMessage.textContent = message;
}

// Show active status
function showActiveStatus(title, message) {
    statusCard.className = 'status-card active';
    statusTitle.textContent = title;
    statusMessage.textContent = message;
}

// Show error status
function showErrorStatus(title, message) {
    statusCard.className = 'status-card error';
    statusTitle.textContent = title;
    statusMessage.textContent = message;
}



// Show action buttons
function showButtons() {
    clearBtn.style.display = 'block';
    exportBtn.style.display = 'block';
}

// Hide action buttons
function hideButtons() {
    clearBtn.style.display = 'none';
    exportBtn.style.display = 'none';
}

// Check if we have user identity
async function checkUserIdentity() {
    // Check storage first
    const result = await chrome.storage.local.get(['LastUserCallData']);
    if (result.LastUserCallData) {
        try {
            const userData = JSON.parse(result.LastUserCallData);
            if (userData && userData.email) {
                return; // We have data
            }
        } catch (e) {
            // Invalid JSON
        }
    }

    // Try to get from Chrome Identity
    chrome.identity.getProfileUserInfo(async (userInfo) => {
        if (userInfo && userInfo.email) {
            // We got it automatically
            await chrome.storage.local.set({
                'LastUserCallData': JSON.stringify(userInfo)
            });
        } else {
            // Need manual input
            showEmailModal();
        }
    });
}

function showEmailModal() {
    emailModal.classList.add('show');
}

function hideEmailModal() {
    emailModal.classList.remove('show');
}

async function saveUserEmail() {
    const email = userEmailInput.value.trim();
    if (!email) {
        alert('Please enter a valid email address');
        return;
    }

    if (!email.includes('@')) {
        alert('Please enter a valid email address');
        return;
    }

    const userData = {
        email: email,
        id: 'manual-' + Date.now()
    };

    await chrome.storage.local.set({
        'LastUserCallData': JSON.stringify(userData)
    });

    hideEmailModal();
}

// Setup event listeners
function setupEventListeners() {
    // Toggle export options
    toggleOptions.addEventListener('click', () => {
        optionsExpanded = !optionsExpanded;
        optionsGrid.style.display = optionsExpanded ? 'grid' : 'none';
        toggleOptions.classList.toggle('open', optionsExpanded);
    });

    // Save Email Button
    saveEmailBtn.addEventListener('click', async () => {
        await saveUserEmail();
    });

    // Clear button
    clearBtn.addEventListener('click', async () => {
        if (confirm('Are you sure you want to clear all extracted data?')) {
            await clearData();
        }
    });

    // Export button
    exportBtn.addEventListener('click', async () => {
        await exportData();
    });


}

// Start monitoring status
function startStatusMonitoring() {
    setInterval(async () => {
        await checkStatus();
    }, 1000);
}

// Clear all data
async function clearData() {
    await chrome.storage.local.remove(['bg_businesses_temp']);
    await chrome.storage.local.set({ 'TF': 0 });

    try {
        await chrome.runtime.sendMessage({ type: 'delete' });
    } catch (error) {

    }


    hideButtons();
    exportOptions.style.display = 'none';
    showReadyStatus('Data Cleared', 'Ready to extract new data');
}

// Export data to CSV
async function exportData() {
    const result = await chrome.storage.local.get(['bg_businesses_temp', 'LastUserCallData']);
    let businesses = result.bg_businesses_temp ? JSON.parse(result.bg_businesses_temp) : [];
    const userData = result.LastUserCallData ? JSON.parse(result.LastUserCallData) : null;

    if (businesses.length === 0) {
        alert('No data to export');
        return;
    }

    // Check subscription status
    let isSubscribed = false;
    if (userData && userData.email) {
        try {
            const response = await fetch('https://api.leads-extractor.com/verify-subscription', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({ email: userData.email })
            });
            const data = await response.json();
            isSubscribed = data.active === true;
        } catch (error) {
            console.error('Error checking subscription:', error);
            // Default to free if check fails
        }
    }

    // Apply limit for free users
    if (!isSubscribed) {
        if (businesses.length > 20) {
            alert('Free version is limited to 20 records. Please upgrade to export all data.');
            businesses = businesses.slice(0, 20);
        }
    }

    // Filter data based on export fields
    const filteredData = businesses.map(business => {
        const filtered = {};
        for (const [key, include] of Object.entries(exportFields)) {
            if (include && business[key] !== undefined) {
                filtered[key] = business[key];
            }
        }
        return filtered;
    });

    // Convert to CSV
    const csv = convertToCSV(filteredData);

    // Download
    downloadCSV(csv, 'google_maps_leads.csv');

    showActiveStatus('Export Complete', `${businesses.length} businesses exported successfully`);
}

// Convert array of objects to CSV
function convertToCSV(data) {
    if (!data || data.length === 0) return '';

    const headers = Object.keys(exportFields).filter(key => exportFields[key]);
    let csv = headers.join(',') + '\n';

    data.forEach(row => {
        const values = headers.map(header => {
            const value = row[header] !== undefined ? row[header] : '';
            // Escape quotes and wrap in quotes if contains comma or quote
            const stringValue = String(value).replace(/"/g, '""');
            return /[,"\n]/.test(stringValue) ? `"${stringValue}"` : stringValue;
        });
        csv += values.join(',') + '\n';
    });

    return csv;
}

// Download CSV file
function downloadCSV(csv, filename) {
    const blob = new Blob(['\ufeff' + csv], { type: 'text/csv;charset=utf-8;' });
    const url = URL.createObjectURL(blob);
    const link = document.createElement('a');
    link.setAttribute('href', url);
    link.setAttribute('download', filename);
    link.style.display = 'none';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
    URL.revokeObjectURL(url);
}

// Extract social media from HTML
function extractSocialMedia(html, url) {
    const parser = new DOMParser();
    const doc = parser.parseFromString(html, 'text/html');

    const getLink = (selector) => {
        const links = doc.querySelectorAll(selector);
        return links.length > 0 ? links[0].href : '';
    };

    const getValue = (pattern) => {
        return html.includes(pattern) ? 'Yes' : 'No';
    };

    let email = '';
    const emailLinks = doc.querySelectorAll('a[href*="mailto"]');
    if (emailLinks.length > 0) {
        email = decodeURIComponent(emailLinks[0].href.replace('mailto:', ''));
        email = email.split('?')[0]; // Remove query parameters
    }

    return {
        'email': email,
        'linkedin_url': getLink('a[href*="linkedin"]'),
        'facebook_url': getLink('a[href*="facebook"]'),
        'twitter_url': getLink('a[href*="twitter"]'),
        'instagram_url': getLink('a[href*="instagram"]'),
        'youtube_url': getLink('a[href*="youtube"]'),
        'pinterest_url': getLink('a[href*="pinterest"]'),
        'facebook pixel': getValue('fbq'),
        'google tag manager': getValue('gtag'),
        'google analytics': getValue('_gaq'),
        'checked': true
    };
}
